<?php
namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Vendor;
use App\Models\VendorBranch;
use App\Models\Zone;
use App\Services\GeoService;
use App\Services\PricingService;
use App\Services\BranchService;

class OrderQuoteController extends Controller
{
    public function quote(Request $request)
    {
        $data = $request->validate([
            'vendor_id' => 'required|integer|exists:vendors,id',
            'dropoff_lat' => 'required|numeric|between:-90,90',
            'dropoff_lng' => 'required|numeric|between:-180,180',
            'type' => 'nullable|string|in:delivery,grocery,courier',
        ]);

        $geo = app(GeoService::class);
        $zones = Zone::where('is_active', true)->get();

        $hitZone = null;
        foreach ($zones as $z) {
            $poly = $z->polygon ?? [];
            if ($poly && $geo->pointInPolygon((float)$data['dropoff_lat'], (float)$data['dropoff_lng'], $poly)) {
                $hitZone = $z; break;
            }
        }
        if (!$hitZone) return response()->json(['ok'=>false,'message'=>'Outside service area'], 422);

        $vendor = Vendor::findOrFail($data['vendor_id']);

        $branches = VendorBranch::where('vendor_id',$vendor->id)
            ->where('is_active',true)
            ->where('zone_id', $hitZone->id)
            ->with('hours')->get();

        $branchService = app(BranchService::class);
        $pricing = app(PricingService::class);

        $best=null; $bestKm=999999;
        foreach ($branches as $b) {
            if(!$branchService->isOpenNow($b)) continue;
            $km = $pricing->haversineKm((float)$b->lat,(float)$b->lng,(float)$data['dropoff_lat'],(float)$data['dropoff_lng']);
            if($km < $bestKm){ $bestKm=$km; $best=$b; }
        }
        if(!$best) return response()->json(['ok'=>false,'message'=>'No open branch'], 422);

        $type = $data['type'] ?? 'delivery';
        $calc = $pricing->calculate($type, $bestKm);
        if(!$calc['ok']) return response()->json(['ok'=>false,'message'=>$calc['message']], 422);

        return response()->json([
            'ok'=>true,
            'zone'=>['id'=>$hitZone->id,'name'=>$hitZone->name],
            'branch'=>['id'=>$best->id,'name'=>$best->name],
            'distance_km'=>round($bestKm,2),
            'delivery_fee'=>$calc['delivery_fee'],
            'pricing_meta'=>$calc['meta'],
        ]);
    }
}
