<?php
namespace App\Http\Controllers;

use App\Models\Ticket;
use App\Models\TicketMessage;
use App\Services\NotificationService;
use App\Events\TicketMessageCreated;
use Illuminate\Http\Request;

class TicketController extends Controller
{
    public function create(Request $request)
    {
        $user = $request->user();
        $data = $request->validate([
            'subject' => 'required|string|max:200',
            'message' => 'required|string|max:5000',
            'category' => 'nullable|string|max:50',
            'priority' => 'nullable|string|in:low,normal,high,urgent',
            'order_id' => 'nullable|integer',
        ]);

        $ticket = Ticket::create([
            'user_id'=>$user->id,
            'subject'=>$data['subject'],
            'category'=>$data['category'] ?? null,
            'priority'=>$data['priority'] ?? 'normal',
            'order_id'=>$data['order_id'] ?? null,
            'status'=>'open',
        ]);

        $msg = TicketMessage::create([
            'ticket_id'=>$ticket->id,
            'sender_id'=>$user->id,
            'message'=>$data['message'],
        ]);

        event(new TicketMessageCreated($ticket->id,$user->id,$msg->message,$msg->created_at->toISOString()));

        $notify = app(NotificationService::class);
        $admins = \App\Models\User::where('role','admin')->pluck('id')->all();
        foreach ($admins as $adminId) {
            $notify->toUser($adminId,'تذكرة دعم جديدة',"Ticket #{$ticket->id}: {$ticket->subject}",[
                'type'=>'ticket','ticket_id'=>(string)$ticket->id
            ]);
        }

        return response()->json(['ticket'=>$ticket,'first_message'=>$msg], 201);
    }

    public function mine(Request $request)
    {
        return Ticket::where('user_id',$request->user()->id)->latest()->get();
    }

    public function show(Request $request, Ticket $ticket)
    {
        $user = $request->user();
        if ((int)$ticket->user_id !== (int)$user->id && $user->role !== 'admin') {
            return response()->json(['message'=>'Forbidden'], 403);
        }
        $ticket->load(['messages'=>fn($q)=>$q->orderBy('id','asc')]);
        return response()->json($ticket);
    }

    public function reply(Request $request, Ticket $ticket)
    {
        $user = $request->user();
        if ((int)$ticket->user_id !== (int)$user->id && $user->role !== 'admin') {
            return response()->json(['message'=>'Forbidden'], 403);
        }

        $data = $request->validate(['message'=>'required|string|max:5000']);

        if ($user->role !== 'admin' && in_array($ticket->status,['resolved','closed'],true)) {
            $ticket->status = 'open';
            $ticket->save();
        }

        $msg = TicketMessage::create([
            'ticket_id'=>$ticket->id,
            'sender_id'=>$user->id,
            'message'=>$data['message'],
        ]);

        event(new TicketMessageCreated($ticket->id,$user->id,$msg->message,$msg->created_at->toISOString()));

        $notify = app(NotificationService::class);
        if ($user->role === 'admin') {
            $notify->toUser($ticket->user_id,'رد من الدعم',$ticket->subject,['type'=>'ticket','ticket_id'=>(string)$ticket->id]);
        } else {
            $targets = $ticket->assigned_admin_id ? [$ticket->assigned_admin_id] : \App\Models\User::where('role','admin')->pluck('id')->all();
            foreach ($targets as $adminId) {
                $notify->toUser($adminId,'تحديث تذكرة دعم',"Ticket #{$ticket->id}: {$ticket->subject}",['type'=>'ticket','ticket_id'=>(string)$ticket->id]);
            }
        }

        return response()->json($msg, 201);
    }
}
