<?php
namespace App\Services;

class PricingService
{
    public function haversineKm(float $lat1, float $lon1, float $lat2, float $lon2): float
    {
        $R=6371;
        $dLat=deg2rad($lat2-$lat1); $dLon=deg2rad($lon2-$lon1);
        $a=sin($dLat/2)**2 + cos(deg2rad($lat1))*cos(deg2rad($lat2))*sin($dLon/2)**2;
        return $R*(2*atan2(sqrt($a), sqrt(1-$a)));
    }

    public function rules(string $type): array
    {
        return match ($type) {
            'grocery' => ['base'=>9,'perKm'=>2.2,'min'=>12,'max'=>35,'maxKm'=>12],
            'courier' => ['base'=>12,'perKm'=>2.5,'min'=>15,'max'=>45,'maxKm'=>18],
            default   => ['base'=>8,'perKm'=>2.0,'min'=>10,'max'=>30,'maxKm'=>10],
        };
    }

    public function surgeMultiplier(): float
    {
        $h = (int) now()->format('H');
        $isPeak = ($h >= 12 && $h < 15) || ($h >= 19 && $h < 23);
        return $isPeak ? 1.25 : 1.0;
    }

    public function calculate(string $type, float $km): array
    {
        $r = $this->rules($type);
        if ($km > $r['maxKm']) return ['ok'=>false,'message'=>"Distance exceeds limit ({$r['maxKm']} km)"];

        $raw = $r['base'] + ($r['perKm'] * $km);
        $surge = $this->surgeMultiplier();
        $fee = max($r['min'], min($r['max'], round($raw * $surge, 2)));

        return ['ok'=>true,'delivery_fee'=>$fee,'meta'=>[
            'base'=>$r['base'],'per_km'=>$r['perKm'],'surge'=>$surge,'raw'=>round($raw,2),
            'min'=>$r['min'],'max'=>$r['max'],'max_km'=>$r['maxKm'],
        ]];
    }
}
