import Link from "next/link";
import { useEffect, useState } from "react";
import { api } from "../src/lib/api";
import { initPushAndGetToken } from "../src/lib/firebase";

type Order = { id:number; status:string; total:number };

export default function VendorHome() {
  const [orders, setOrders] = useState<Order[]>([]);
  const [msg, setMsg] = useState("");

  const load = async () => {
    try { setOrders(await api<Order[]>("/api/vendor/orders")); } catch (e:any) { setMsg(e.message); }
  };

  useEffect(()=>{ load(); }, []);

  const accept = async (id:number) => {
    const res = await api<any>(`/api/vendor/orders/${id}/accept`, { method:"POST" });
    setMsg(res.driver_assigned ? `Accepted. Driver: ${res.driver_id}` : "Accepted. No driver yet.");
    load();
  };

  const enablePush = async () => {
    const token = await initPushAndGetToken();
    if (!token) return setMsg("Push token not available. Check Firebase config / permissions.");
    await api("/api/devices/register", { method:"POST", body: JSON.stringify({ token, platform:"web" }) });
    setMsg("Push enabled ✅");
  };

  return (
    <div style={{ maxWidth: 820, margin:"24px auto", padding:16 }}>
      <div style={{ display:"flex", justifyContent:"space-between" }}>
        <h2>Vendor</h2>
        <div style={{ display:"flex", gap:12 }}>
          <button onClick={enablePush}>Enable Push</button>
          <Link href="/login">Login</Link>
        </div>
      </div>
      {msg && <p>{msg}</p>}
      <button onClick={load} style={{ padding:10 }}>Refresh</button>
      {orders.map(o => (
        <div key={o.id} style={{ border:"1px solid #ddd", padding:12, marginTop:10, borderRadius:10 }}>
          <div>Order #{o.id} — <b>{o.status}</b> — Total: {o.total}</div>
          {['pending','paid'].includes(o.status) && (
            <button onClick={()=>accept(o.id)} style={{ marginTop:8 }}>Accept</button>
          )}
        </div>
      ))}
    </div>
  );
}
